// Author:  Douglas Wilhelm Harder
// Copyright (c) 2009 by Douglas Wilhelm Harder.  All rights reserved.

/***************************************************
 * Add the entries of the vector u onto the
 * entries of this vector.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator+= ( Vector<N, D> const &u ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] += u.array[i];
	}

	return *this;
}

/***************************************************
 * Subtract the entries of the vector u from the
 * entries of this vector.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator-= ( Vector<N, D> const &u ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] -= u.array[i];
	}

	return *this;
}

/***************************************************
 * Multiply the entries of the vector u
 * elementwise by the entries of this vector.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator*= ( Vector<N, D> const &u ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] *= u.array[i];
	}

	return *this;
}

/***************************************************
 * Divide the entries of this vector
 * elementwise by the entries of the vector u.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/


template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator/= ( Vector<N, D> const &u ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] /= u.array[i];
	}

	return *this;
}

/***************************************************************************** 
 * *************************************************************************** 
 * *
 * *   FRIENDS
 * *
 * *************************************************************************** 
 *****************************************************************************/

/***************************************************
 * Add the entries of the vectors u and v.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator+ ( Vector<N, D> const &u, Vector<N, D> const &v ) {
	Vector<N, D> w( u );

	w += v;

	return w;
}

/***************************************************
 * Subtract the entries of the vector v from the
 * entries of the vector u.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator- ( Vector<N, D> const &u, Vector<N, D> const &v ) {
	Vector<N, D> w( u );

	w -= v;

	return w;
}

/***************************************************
 * Multiply the entries of vectors u and v
 * elementwise.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator* ( Vector<N, D> const &u, Vector<N, D> const &v ) {
	Vector<N, D> w( u );

	w *= v;

	return w;
}

/***************************************************
 * Divide the entries of the vector u by the
 * entries of the vector v.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator/ ( Vector<N, D> const &u, Vector<N, D> const &v ) {
	Vector<N, D> w( u );

	w /= v;

	return w;
}

/***************************************************
 * Take the inner product of the row vector u
 * and the column vector v.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N>
double operator* ( Vector<N, ROW> const &u, Vector<N, COLUMN> const &v ) {
	double inner_product = 0.0;

	for ( int i = 0; i < N; ++i ) {
		inner_product += u.array[i] * v.array[i];
	}

	return inner_product;
}
