// Author:  Douglas Wilhelm Harder
// Copyright (c) 2009 by Douglas Wilhelm Harder.  All rights reserved.

/***************************************************
 * Add x to each entry of the vector.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator+= ( double x ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] += x;
	}

	return *this;
}

/***************************************************
 * Subtract x from each entry of the vector.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator-= ( double x ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] -= x;
	}

	return *this;
}

/***************************************************
 * Multiply each entry of the vector by x.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator*= ( double x ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] *= x;
	}

	return *this;
}

/***************************************************
 * Divide each entry of the vector by x.
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  &Vector<N, D>::operator/= ( double x ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] /= x;
	}

	return *this;
}

/***************************************************************************** 
 * *************************************************************************** 
 * *
 * *   FRIENDS
 * *
 * *************************************************************************** 
 *****************************************************************************/

/***************************************************
 * Return x added to each entry of the vector u.
 *  - See operator+= above.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator+ ( Vector<N, D> const &u, double x ) {
	Vector<N, D> v( u );

	v += x;

	return v;
}

/***************************************************
 * Return x subtracted from each entry of the
 * vector u.
 *  - See operator-= above.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator- ( Vector<N, D> const &u, double x ) {
	Vector<N, D> v( u );

	v -= x;

	return v;
}

/***************************************************
 * Return each entry of the vector u
 * multiplied by x.
 *  - See operator*= above.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator* ( Vector<N, D> const &u, double x ) {
	Vector<N, D> v( u );

	v *= x;

	return v;
}

/***************************************************
 * Return each entry of the vector u
 * divided by x.
 *  - See operator/= above.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator/ ( Vector<N, D> const &u, double x ) {
	Vector<N, D> v( u );

	v /= x;

	return v;
}

/***************************************************
 * Return x added to each entry of the vector u.
 *  - See operator+= above.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator+ ( double x, Vector<N, D> const &u ) {
	Vector<N, D> v( u );

	v += x;

	return v;
}

/***************************************************
 * Return each entry of the vector u subtracted
 * from x.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator- ( double x, Vector<N, D> const &u ) {
	Vector<N, D> v;

	for ( int i = 0; i < N; ++i ) {
		v.array[i] = x - u.array[i];
	}

	return v;
}

/***************************************************
 * Return each entry of the vector u
 * multiplied by x.
 *  - See operator*= above.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator* ( double x, Vector<N, D> const &u ) {
	Vector<N, D> v( u );

	v *= x;

	return v;
}

/***************************************************
 * Return each entry of the vector u dividing x.
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>  operator/ ( double x, Vector<N, D> const &u ) {
	Vector<N, D> v;

	for ( int i = 0; i < N; ++i ) {
		v.array[i] = x/u.array[i];
	}

	return v;
}
