// Author:  Douglas Wilhelm Harder
// Copyright (c) 2009 by Douglas Wilhelm Harder.  All rights reserved.

#include <cstdlib>
#include <cassert>

/***************************************************
 * Constructor
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>::Vector( double x ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] = x;
	}
}

/***************************************************
 * Copy Constructor
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D>::Vector( Vector<N, D> const &v ) {
	for ( int i = 0; i < N; ++i ) {
		array[i] = v.array[i];
	}
}

/***************************************************
 * Assignment Operator
 *
 * Run time:   O( N )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D> &Vector<N, D>::operator= ( Vector<N, D> const &v ) {
	if ( this == &v ) {
		return *this;
	}

	for ( int i = 0; i < N; ++i ) {
		array[i] = v.array[i];
	}

	return *this;
}

/***************************************************
 * Indexing Operator
 * Return a reference to the ith entry
 * of the vector:
 *     0    1     2    ...   N-2  N-1
 *    -N  -N+1  -N+2   ...   -2   -1
 *
 * Run time:   O( 1 )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
double &Vector<N, D>::operator() ( int i ) {
	assert( i >= -N && i < N );

	return i >= 0 ? array[i] : array[N + i];
}

/***************************************************
 * Indexing Operator
 * Return the value of the ith entry
 * of the vector:
 *     0    1     2    ...   N-2  N-1
 *    -N  -N+1  -N+2   ...   -2   -1
 *
 * Run time:   O( 1 )
 * Memory:     O( 1 )
 ***************************************************/

template<int N, Orientation D>
double Vector<N, D>::operator() ( int i ) const {
	assert( i >= -N && i < N );

	return i >= 0 ? array[i] : array[N + i];
}

/***************************************************
 * Construct a random vector with entries on the
 * open range (0, 1)
 *
 * Run time:   O( N )
 * Memory:     O( N )
 ***************************************************/

template<int N, Orientation D>
Vector<N, D> Vector<N, D>::random() {
	Vector<N, D> v;

	for ( int i = 0; i < N; ++i ) {
		v.array[i] = static_cast<double>( std::rand() + 1 ) / static_cast<double>( RAND_MAX + 2 );
	}

	return v;
}
