// Author:  Douglas Wilhelm Harder
// Copyright (c) 2009 by Douglas Wilhelm Harder.  All rights reserved.

#include <cmath>

/***************************************************
 * The infinity norm of the matrix:
 *  - The maximum absolute entry.
 *
 * Run time:   O(N)
 * Memory:     O(1)
 ***************************************************/

template<int N, Orientation D>
double Vector<N, D>::norm() const {
	double value = 0;

	for ( int i = 0; i < N; ++i ) {
		value = std::max( std::fabs( array[i] ), value );
	}

	return value;
}

template<int N, Orientation D>
double norm( Vector<N, D> const &v ) {
	return v.norm();
}

/***************************************************
 * The 1- and 2-norms of the matrix.
 *  - The sum of the absolute values of the
 *    entries, and
 *  - The square root of the sum of the squares
 *    of the entries.
 *
 * Run time:   O(N)
 * Memory:     O(1)
 ***************************************************/

template<int N, Orientation D>
double Vector<N, D>::norm( int n ) const {
	double value = 0;

	if ( n == 1 ) {
		for ( int i = 0; i < N; ++i ) {
			value += std::fabs( array[i] );
		}

		return value;
	} else if ( n == 2 ) {
		for ( int i = 0; i < N; ++i ) {
			value += array[i]*array[i];
		}

		return std::sqrt( value );
	}

	throw invalid_norm( n );
}

template<int N, Orientation D>
double norm( Vector<N, D> const &v, int n ) {
	return v.norm( n );
}

/***************************************************
 * The transpose of a column vector is a row
 * vector with the same entries.
 *
 * Run time:   O(N)
 * Memory:     O(N)
 ***************************************************/

template<int N>
Vector<N, ROW> transpose( Vector<N, COLUMN> const &u ) {
	Vector<N, ROW> v;

	for ( int i = 0; i < N; ++i ) {
		v.array[i] = u.array[i];
	}

	return v;
}

/***************************************************
 * The transpose of a row vector is a column
 * vector with the same entries.
 *
 * Run time:   O(N)
 * Memory:     O(N)
 ***************************************************/

template<int N>
Vector<N, COLUMN> transpose( Vector<N, ROW> const &u ) {
	Vector<N, COLUMN> v;

	for ( int i = 0; i < N; ++i ) {
		v.array[i] = u.array[i];
	}

	return v;
}
