/******************************************
 * C++ Support for the Number Systems Package
 * Version: 1.0.9
 * Author:  Douglas Wilhelm Harder
 * Date:    2008/03/03
 *
 * Copyright (c) 2006-2008 by Douglas Wilhelm Harder.
 * All rights reserved.
 ******************************************/

#include <iostream>
#include <cmath>
#include <string>
#include <limits>
#include "Support.h"

template <typename T>
T Support<T>::Gamma( int n ) {
	return ( n <= 0 || n >= 172 ) ? POS_INF : GAMMA_TAB[n - 1];
}

template <typename T>
T Support<T>::sec( T x ) {
	return 1.0 / std::cos(x);
}

template <typename T>
T Support<T>::csc( T x ) {
	return 1.0 / std::sin(x);
}

template <typename T>
T Support<T>::cot( T x ) {
	return 1.0 / std::tan(x);
}

template <typename T>
T Support<T>::sech( T x ) {
	return 1.0 / std::cosh(x);
}

template <typename T>
T Support<T>::csch( T x ) {
	return 1.0 / std::sinh(x);
}

template <typename T>
T Support<T>::coth( T x ) {
	return 1.0 / std::tanh(x);
}

template <typename T>
T Support<T>::sign( T x ) {
	unsigned char *chr = reinterpret_cast<unsigned char *>( &x );

	return bigendian ?
		( (chr[sizeof( T ) - 1] & 0x80) ? -1.0 : 1.0 ) :
		( (chr[0] & 0x80) ? -1.0 : 1.0 ) ;
}

template <typename T>
bool Support<T>::is_pos_inf( T x ) {
	return x == POS_INF;
}

template <typename T>
bool Support<T>::is_neg_inf( T x ) {
	return x == NEG_INF;
}

template <typename T>
bool Support<T>::is_pos_zero( T x ) {
	return (x == 0) && sign( x ) == 1.0;
}

template <typename T>
bool Support<T>::is_neg_zero( T x ) {
	return (x == 0) && sign( x ) == -1.0;
}

template <typename T>
bool Support<T>::is_inf( T x ) {
	return ( x == POS_INF ) || ( x == NEG_INF );
}

template <typename T>
bool Support<T>::is_nan( T x ) {
	return x != x;
}

/******************************************
* IO Stream Functions
******************************************/

template <typename T> void Support<T>::print_real( T r, std::ostream & out ) {
	if ( is_pos_zero( r ) ) {
		out << "0";
	} else if ( is_neg_zero( r ) ) {
		out << "-0";
	} else if ( is_nan( r ) ) {
		out << "NaN";
	} else {
		out << r;
	}
}

template <typename T> void Support<T>::print_imaginary( T i, const std::string & s, std::ostream & out ) {
	if ( is_pos_zero( i ) ) {
		out << " + 0" << s;
	} else if ( is_neg_zero( i ) ) {
		out << " - 0" << s;
	} else if ( is_nan( i ) ) {
		out << " + NaN" << s;
	} else if ( i == 1.0 ) {
		out << " + " << s;
	} else if ( i == -1.0 ) {
		out << " - " << s;
	} else if ( i > 0 ) {
		out << " + " << i << s;
	} else {
		out << " - " << (-i) << s;
	}
}

template <typename T> void Support<T>::print_imaginary( T i, char c, std::ostream & out ) {
	if ( is_pos_zero( i ) ) {
		out << " + 0" << c;
	} else if ( is_neg_zero( i ) ) {
		out << " - 0" << c;
	} else if ( is_nan( i ) ) {
		out << " + NaN" << c;
	} else if ( i == 1.0 ) {
		out << " + " << c;
	} else if ( i == -1.0 ) {
		out << " - " << c;
	} else if ( i > 0 ) {
		out << " + " << i << c;
	} else {
		out << " - " << (-i) << c;
	}
}

static bool is_bigendian() {
	double d = 1.0;

	unsigned char *chr = reinterpret_cast<unsigned char *>( &d );

	return ( chr[0] == 0 );
}

template <typename T>
short Support<T>::bigendian = is_bigendian();

template class Support<double>;

template <> const double Support<double>::POS_INF =  std::numeric_limits<double>::infinity();
template <> const double Support<double>::NEG_INF = -std::numeric_limits<double>::infinity();
template <> const double Support<double>::NaN = std::log(-1.0);
template <> const double Support<double>::PI =  3.1415926535897932385;   // evalf[20](Pi)
template <> const double Support<double>::PI2 = 1.5707963267948966192;   // evalf[20](Pi/2)
template <> const double Support<double>::GAMMA_TAB[171] = {
	1.0, 1.0, 2.0, 6.0, 24.0, 120.0, 720.0, 5040.0, 40320.0,
	362880.0, 3628800.0, 39916800.0, 479001600.0, 6227020800.0,
	87178291200.0, 1307674368000.0, 20922789888000.0,
	355687428096000.0, 6402373705728000.0, 121645100408832000.0,
	2432902008176640000.0, 51090942171709440000.0,
	0.11240007277776076800e22, .25852016738884976640e23,
	.62044840173323943936e24, .15511210043330985984e26,
	.40329146112660563558e27, .10888869450418352161e29,
	.30488834461171386050e30, .88417619937397019545e31,
	.26525285981219105864e33, .82228386541779228177e34,
	.26313083693369353017e36, .86833176188118864955e37,
	.29523279903960414085e39, .10333147966386144930e41,
	.37199332678990121747e42, .13763753091226345046e44,
	.52302261746660111176e45, .20397882081197443359e47,
	.81591528324789773435e48, .33452526613163807108e50,
	.14050061177528798985e52, .60415263063373835637e53,
	.26582715747884487680e55, .11962222086548019456e57,
	.55026221598120889499e58, .25862324151116818064e60,
	.12413915592536072671e62, .60828186403426756087e63,
	.30414093201713378044e65, .15511187532873822802e67,
	.80658175170943878572e68, .42748832840600255643e70,
	.23084369733924138047e72, .12696403353658275926e74,
	.71099858780486345185e75, .40526919504877216756e77,
	.23505613312828785718e79, .13868311854568983574e81,
	.83209871127413901443e82, .50758021387722479880e84,
	.31469973260387937526e86, .19826083154044400641e88,
	.12688693218588416410e90, .82476505920824706667e91,
	.54434493907744306400e93, .36471110918188685288e95,
	.24800355424368305996e97, .17112245242814131137e99,
	.11978571669969891796e101, .85047858856786231752e102,
	.61234458376886086862e104, .44701154615126843409e106,
	.33078854415193864123e108, .24809140811395398092e110,
	.18854947016660502550e112, .14518309202828586963e114,
	.11324281178206297831e116, .89461821307829752869e117,
	.71569457046263802295e119, .57971260207473679859e121,
	.47536433370128417484e123, .39455239697206586512e125,
	.33142401345653532670e127, .28171041143805502769e129,
	.24227095383672732382e131, .21077572983795277172e133,
	.18548264225739843911e135, .16507955160908461081e137,
	.14857159644817614973e139, .13520015276784029626e141,
	.12438414054641307255e143, .11567725070816415748e145,
	.10873661566567430803e147, .10329978488239059263e149,
	.99167793487094968921e150, .96192759682482119853e152,
	.94268904488832477456e154, .93326215443944152682e156,
	.93326215443944152682e158, .94259477598383594209e160,
	.96144667150351266093e162, .99029007164861804075e164,
	.10299016745145627624e167, .10813967582402909005e169,
	.11462805637347083545e171, .12265202031961379394e173,
	.13246418194518289745e175, .14438595832024935822e177,
	.15882455415227429404e179, .17629525510902446639e181,
	.19745068572210740235e183, .22311927486598136466e185,
	.25435597334721875571e187, .29250936934930156907e189,
	.33931086844518982012e191, .39699371608087208954e193,
	.46845258497542906566e195, .55745857612076058813e197,
	.66895029134491270576e199, .80942985252734437397e201,
	.98750442008336013624e203, .12146304367025329676e206,
	.15061417415111408798e208, .18826771768889260997e210,
	.23721732428800468857e212, .30126600184576595448e214,
	.38562048236258042174e216, .49745042224772874404e218,
	.64668554892204736725e220, .84715806908788205110e222,
	.11182486511960043074e225, .14872707060906857289e227,
	.19929427461615188767e229, .26904727073180504836e231,
	.36590428819525486577e233, .50128887482749916610e235,
	.69177864726194884922e237, .96157231969410890042e239,
	.13462012475717524606e242, .18981437590761709694e244,
	.26953641378881627766e246, .38543707171800727705e248,
	.55502938327393047896e250, .80479260574719919448e252,
	.11749972043909108239e255, .17272458904546389112e257,
	.25563239178728655886e259, .38089226376305697270e261,
	.57133839564458545905e263, .86272097742332404316e265,
	.13113358856834525456e268, .20063439050956823948e270,
	.30897696138473508880e272, .47891429014633938763e274,
	.74710629262828944471e276, .11729568794264144282e279,
	.18532718694937347965e281, .29467022724950383265e283,
	.47147236359920613224e285, .75907050539472187291e287,
	.12296942187394494341e290, .20044015765453025776e292,
	.32872185855342962273e294, .54239106661315887750e296,
	.90036917057784373665e298, .15036165148649990402e301,
	.25260757449731983875e303, .42690680090047052749e305,
	.72574156153079989674e307
};


template class Support<float>;

template <> const float Support<float>::POS_INF =  std::numeric_limits<float>::infinity();
template <> const float Support<float>::NEG_INF = -std::numeric_limits<float>::infinity();
template <> const float Support<float>::NaN = std::log(-1.0);
template <> const float Support<float>::PI =  3.1415926535897932385;   // evalf[20](Pi)
template <> const float Support<float>::PI2 = 1.5707963267948966192;   // evalf[20](Pi/2)
template <> const float Support<float>::GAMMA_TAB[171] = {
	1.0, 1.0, 2.0, 6.0, 24.0, 120.0, 720.0, 5040.0, 40320.0,
	362880.0, 3628800.0, 39916800.0, 479001600.0, 6227020800.0,
	87178291200.0, 1307674368000.0, 20922789888000.0,
	355687428096000.0, 6402373705728000.0, 121645100408832000.0,
	2432902008176640000.0, 51090942171709440000.0,
	0.11240007277776076800e22, .25852016738884976640e23,
	.62044840173323943936e24, .15511210043330985984e26,
	.40329146112660563558e27, .10888869450418352161e29,
	.30488834461171386050e30, .88417619937397019545e31,
	.26525285981219105864e33, .82228386541779228177e34,
	.26313083693369353017e36, .86833176188118864955e37,
	.29523279903960414085e39, .10333147966386144930e41,
	.37199332678990121747e42, .13763753091226345046e44,
	.52302261746660111176e45, .20397882081197443359e47,
	.81591528324789773435e48, .33452526613163807108e50,
	.14050061177528798985e52, .60415263063373835637e53,
	.26582715747884487680e55, .11962222086548019456e57,
	.55026221598120889499e58, .25862324151116818064e60,
	.12413915592536072671e62, .60828186403426756087e63,
	.30414093201713378044e65, .15511187532873822802e67,
	.80658175170943878572e68, .42748832840600255643e70,
	.23084369733924138047e72, .12696403353658275926e74,
	.71099858780486345185e75, .40526919504877216756e77,
	.23505613312828785718e79, .13868311854568983574e81,
	.83209871127413901443e82, .50758021387722479880e84,
	.31469973260387937526e86, .19826083154044400641e88,
	.12688693218588416410e90, .82476505920824706667e91,
	.54434493907744306400e93, .36471110918188685288e95,
	.24800355424368305996e97, .17112245242814131137e99,
	.11978571669969891796e101, .85047858856786231752e102,
	.61234458376886086862e104, .44701154615126843409e106,
	.33078854415193864123e108, .24809140811395398092e110,
	.18854947016660502550e112, .14518309202828586963e114,
	.11324281178206297831e116, .89461821307829752869e117,
	.71569457046263802295e119, .57971260207473679859e121,
	.47536433370128417484e123, .39455239697206586512e125,
	.33142401345653532670e127, .28171041143805502769e129,
	.24227095383672732382e131, .21077572983795277172e133,
	.18548264225739843911e135, .16507955160908461081e137,
	.14857159644817614973e139, .13520015276784029626e141,
	.12438414054641307255e143, .11567725070816415748e145,
	.10873661566567430803e147, .10329978488239059263e149,
	.99167793487094968921e150, .96192759682482119853e152,
	.94268904488832477456e154, .93326215443944152682e156,
	.93326215443944152682e158, .94259477598383594209e160,
	.96144667150351266093e162, .99029007164861804075e164,
	.10299016745145627624e167, .10813967582402909005e169,
	.11462805637347083545e171, .12265202031961379394e173,
	.13246418194518289745e175, .14438595832024935822e177,
	.15882455415227429404e179, .17629525510902446639e181,
	.19745068572210740235e183, .22311927486598136466e185,
	.25435597334721875571e187, .29250936934930156907e189,
	.33931086844518982012e191, .39699371608087208954e193,
	.46845258497542906566e195, .55745857612076058813e197,
	.66895029134491270576e199, .80942985252734437397e201,
	.98750442008336013624e203, .12146304367025329676e206,
	.15061417415111408798e208, .18826771768889260997e210,
	.23721732428800468857e212, .30126600184576595448e214,
	.38562048236258042174e216, .49745042224772874404e218,
	.64668554892204736725e220, .84715806908788205110e222,
	.11182486511960043074e225, .14872707060906857289e227,
	.19929427461615188767e229, .26904727073180504836e231,
	.36590428819525486577e233, .50128887482749916610e235,
	.69177864726194884922e237, .96157231969410890042e239,
	.13462012475717524606e242, .18981437590761709694e244,
	.26953641378881627766e246, .38543707171800727705e248,
	.55502938327393047896e250, .80479260574719919448e252,
	.11749972043909108239e255, .17272458904546389112e257,
	.25563239178728655886e259, .38089226376305697270e261,
	.57133839564458545905e263, .86272097742332404316e265,
	.13113358856834525456e268, .20063439050956823948e270,
	.30897696138473508880e272, .47891429014633938763e274,
	.74710629262828944471e276, .11729568794264144282e279,
	.18532718694937347965e281, .29467022724950383265e283,
	.47147236359920613224e285, .75907050539472187291e287,
	.12296942187394494341e290, .20044015765453025776e292,
	.32872185855342962273e294, .54239106661315887750e296,
	.90036917057784373665e298, .15036165148649990402e301,
	.25260757449731983875e303, .42690680090047052749e305,
	.72574156153079989674e307
};

template class Support<long double>;

template <> const long double Support<long double>::POS_INF =  std::numeric_limits<long double>::infinity();
template <> const long double Support<long double>::NEG_INF = -std::numeric_limits<long double>::infinity();
template <> const long double Support<long double>::NaN = std::log(-1.0);
template <> const long double Support<long double>::PI =  3.1415926535897932385;   // evalf[20](Pi)
template <> const long double Support<long double>::PI2 = 1.5707963267948966192;   // evalf[20](Pi/2)
template <> const long double Support<long double>::GAMMA_TAB[171] = {
	1.0, 1.0, 2.0, 6.0, 24.0, 120.0, 720.0, 5040.0, 40320.0,
	362880.0, 3628800.0, 39916800.0, 479001600.0, 6227020800.0,
	87178291200.0, 1307674368000.0, 20922789888000.0,
	355687428096000.0, 6402373705728000.0, 121645100408832000.0,
	2432902008176640000.0, 51090942171709440000.0,
	0.11240007277776076800e22, .25852016738884976640e23,
	.62044840173323943936e24, .15511210043330985984e26,
	.40329146112660563558e27, .10888869450418352161e29,
	.30488834461171386050e30, .88417619937397019545e31,
	.26525285981219105864e33, .82228386541779228177e34,
	.26313083693369353017e36, .86833176188118864955e37,
	.29523279903960414085e39, .10333147966386144930e41,
	.37199332678990121747e42, .13763753091226345046e44,
	.52302261746660111176e45, .20397882081197443359e47,
	.81591528324789773435e48, .33452526613163807108e50,
	.14050061177528798985e52, .60415263063373835637e53,
	.26582715747884487680e55, .11962222086548019456e57,
	.55026221598120889499e58, .25862324151116818064e60,
	.12413915592536072671e62, .60828186403426756087e63,
	.30414093201713378044e65, .15511187532873822802e67,
	.80658175170943878572e68, .42748832840600255643e70,
	.23084369733924138047e72, .12696403353658275926e74,
	.71099858780486345185e75, .40526919504877216756e77,
	.23505613312828785718e79, .13868311854568983574e81,
	.83209871127413901443e82, .50758021387722479880e84,
	.31469973260387937526e86, .19826083154044400641e88,
	.12688693218588416410e90, .82476505920824706667e91,
	.54434493907744306400e93, .36471110918188685288e95,
	.24800355424368305996e97, .17112245242814131137e99,
	.11978571669969891796e101, .85047858856786231752e102,
	.61234458376886086862e104, .44701154615126843409e106,
	.33078854415193864123e108, .24809140811395398092e110,
	.18854947016660502550e112, .14518309202828586963e114,
	.11324281178206297831e116, .89461821307829752869e117,
	.71569457046263802295e119, .57971260207473679859e121,
	.47536433370128417484e123, .39455239697206586512e125,
	.33142401345653532670e127, .28171041143805502769e129,
	.24227095383672732382e131, .21077572983795277172e133,
	.18548264225739843911e135, .16507955160908461081e137,
	.14857159644817614973e139, .13520015276784029626e141,
	.12438414054641307255e143, .11567725070816415748e145,
	.10873661566567430803e147, .10329978488239059263e149,
	.99167793487094968921e150, .96192759682482119853e152,
	.94268904488832477456e154, .93326215443944152682e156,
	.93326215443944152682e158, .94259477598383594209e160,
	.96144667150351266093e162, .99029007164861804075e164,
	.10299016745145627624e167, .10813967582402909005e169,
	.11462805637347083545e171, .12265202031961379394e173,
	.13246418194518289745e175, .14438595832024935822e177,
	.15882455415227429404e179, .17629525510902446639e181,
	.19745068572210740235e183, .22311927486598136466e185,
	.25435597334721875571e187, .29250936934930156907e189,
	.33931086844518982012e191, .39699371608087208954e193,
	.46845258497542906566e195, .55745857612076058813e197,
	.66895029134491270576e199, .80942985252734437397e201,
	.98750442008336013624e203, .12146304367025329676e206,
	.15061417415111408798e208, .18826771768889260997e210,
	.23721732428800468857e212, .30126600184576595448e214,
	.38562048236258042174e216, .49745042224772874404e218,
	.64668554892204736725e220, .84715806908788205110e222,
	.11182486511960043074e225, .14872707060906857289e227,
	.19929427461615188767e229, .26904727073180504836e231,
	.36590428819525486577e233, .50128887482749916610e235,
	.69177864726194884922e237, .96157231969410890042e239,
	.13462012475717524606e242, .18981437590761709694e244,
	.26953641378881627766e246, .38543707171800727705e248,
	.55502938327393047896e250, .80479260574719919448e252,
	.11749972043909108239e255, .17272458904546389112e257,
	.25563239178728655886e259, .38089226376305697270e261,
	.57133839564458545905e263, .86272097742332404316e265,
	.13113358856834525456e268, .20063439050956823948e270,
	.30897696138473508880e272, .47891429014633938763e274,
	.74710629262828944471e276, .11729568794264144282e279,
	.18532718694937347965e281, .29467022724950383265e283,
	.47147236359920613224e285, .75907050539472187291e287,
	.12296942187394494341e290, .20044015765453025776e292,
	.32872185855342962273e294, .54239106661315887750e296,
	.90036917057784373665e298, .15036165148649990402e301,
	.25260757449731983875e303, .42690680090047052749e305,
	.72574156153079989674e307
};

